'use strict';

/* --------------------------------------------------------------
 fallback.js 2016-09-08
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.fallback = jse.libs.fallback || {};

/**
 * ## Fallback Library
 *
 * This library contains a set of deprecated functions that are still present for fallback support. Do not
 * use these methods in new modules.
 *
 * @module JSE/Libs/fallback
 * @exports jse.libs.fallback
 */
(function (exports) {

    'use strict';

    /**
     * Add ":attr" pseudo selector.
     *
     * This pseudo selector is normally enabled by including the JSEngine "jquery_extensions" library. Honeygrid
     * through needs this pseudo selector in this library which might be loaded prior to jquery_extensions and
     * this is why we define it once again in this file.
     */

    if ($.expr.pseudos.attr === undefined) {
        $.expr.pseudos.attr = $.expr.createPseudo(function (selector) {
            var regexp = new RegExp(selector);
            return function (elem) {
                for (var i = 0; i < elem.attributes.length; i++) {
                    var attr = elem.attributes[i];
                    if (regexp.test(attr.name)) {
                        return true;
                    }
                }
                return false;
            };
        });
    }

    /**
     * Add a fallback usage warning in the console.
     *
     * As the JS engine evolves many old features will need to be changed in order to let a finer and clearer
     * API for the JS Engine core mechanisms. Use this method to create a fallback usage warning for the functions
     * placed within this library.
     *
     * @param {String} functionName The deprecated function name.
     *
     * @private
     */
    function _warn(functionName) {
        jse.core.debug.warn('jse.libs.fallback.' + functionName + ' was called! ' + 'Avoid the use of fallback methods in new modules.');
    }

    /**
     * Get the module related data of the provided element.
     *
     * @param {jQuery} $element
     * @param {String} moduleName
     *
     * @return {Object}
     */
    exports._data = function ($element, moduleName) {
        _warn('_data');

        var initialData = $element.data(),
            filteredData = {};

        // Searches for module relevant data inside the main-data-object.
        // Data for other widgets will not get passed to this widget
        $.each(initialData, function (key, value) {
            if (key.indexOf(moduleName) === 0 || key.indexOf(moduleName.toLowerCase()) === 0) {
                var newKey = key.substr(moduleName.length);
                newKey = newKey.substr(0, 1).toLowerCase() + newKey.substr(1);
                filteredData[newKey] = value;
            }
        });

        return filteredData;
    };

    /**
     * Setup Widget Attribute
     *
     * @param {Object} $element Change the widget attribute of an element.
     */
    exports.setupWidgetAttr = function ($element) {
        _warn('setupWidgetAttr');

        $element.filter(':attr(^data-gx-_), :attr(^data-gambio-_), :attr(^data-jse-_)').add($element.find(':attr(^data-gx-_), :attr(^data-gambio-_), :attr(^data-jse-_)')).each(function () {
            var $self = $(this),
                attributes = $self[0].attributes,
                matchedAttribute = void 0,
                namespaceName = void 0;

            $.each(attributes, function (index, attribute) {
                if (attribute === undefined) {
                    return true; // wrong attribute, continue loop
                }

                matchedAttribute = attribute.name.match(/data-(gambio|gx|jse)-_.*/g);

                if (matchedAttribute !== null && matchedAttribute.length > 0) {
                    namespaceName = matchedAttribute[0].match(/(gambio|gx|jse)/g)[0];

                    $self.attr(attribute.name.replace('data-' + namespaceName + '-_', 'data-' + namespaceName + '-'), attribute.value);
                }
            });
        });
    };

    /**
     * Get URL parameters.
     *
     * @param {String} url
     * @param {Boolean} deep
     *
     * @return {Object}
     */
    exports.getUrlParams = function (url, deep) {
        _warn('getUrlParams');

        url = decodeURIComponent(url || location.href);

        var splitUrl = url.split('?'),
            splitParam = splitUrl.length > 1 ? splitUrl[1].split('&') : [],
            regex = new RegExp(/\[(.*?)\]/g),
            result = {};

        $.each(splitParam, function (i, v) {
            var keyValue = v.split('='),
                regexResult = regex.exec(keyValue[0]),
                base = null,
                basename = keyValue[0].substring(0, keyValue[0].search('\\[')),
                keys = [],
                lastKey = null;

            if (!deep || regexResult === null) {
                result[keyValue[0]] = keyValue[1].split('#')[0];
            } else {

                result[basename] = result[basename] || [];
                base = result[basename];

                do {
                    keys.push(regexResult[1]);
                    regexResult = regex.exec(keyValue[0]);
                } while (regexResult !== null);

                $.each(keys, function (i, v) {
                    var next = keys[i + 1];
                    v = v || '0';

                    if (typeof next === 'string') {
                        base[v] = base[v] || [];
                        base = base[v];
                    } else {
                        base[v] = base[v] || undefined;
                        lastKey = v;
                    }
                });

                if (lastKey !== null) {
                    base[lastKey] = keyValue[1];
                } else {
                    base = keyValue[1];
                }
            }
        });

        return result;
    };

    /**
     * Fallback getData method.
     *
     * This method was included in v1.0 of JS Engine and is replaced by the
     * "jse.libs.form.getData" method.
     *
     * @param {Object} $form Selector of the form to be parsed.
     * @param {String} ignore (optional) jQuery selector string of form elements to be ignored.
     *
     * @return {Object} Returns the data of the form as an object.
     */
    exports.getData = function ($form, ignore) {
        _warn('getData');

        var $elements = $form.find('input, textarea, select'),
            result = {};

        if (ignore) {
            $elements = $elements.filter(':not(' + ignore + ')');
        }

        $elements.each(function () {
            var $self = $(this),
                type = $self.prop('tagName').toLowerCase(),
                name = $self.attr('name'),
                $selected = null;

            type = type !== 'input' ? type : $self.attr('type').toLowerCase();

            switch (type) {
                case 'radio':
                    $form.find('input[name="' + name + '"]:checked').val();
                    break;
                case 'checkbox':
                    if (name.search('\\[') !== -1) {
                        if ($self.prop('checked')) {
                            name = name.substring(0, name.search('\\['));
                            if (result[name] === undefined) {
                                result[name] = [];
                            }
                            result[name].push($(this).val());
                        }
                    } else {
                        result[name] = $self.prop('checked');
                    }
                    break;
                case 'select':
                    $selected = $self.find(':selected');
                    if ($selected.length > 1) {
                        result[name] = [];
                        $selected.each(function () {
                            result[name].push($(this).val());
                        });
                    } else {
                        result[name] = $selected.val();
                    }
                    break;
                case 'button':
                    break;
                default:
                    if (name) {
                        result[name] = $self.val();
                    }
                    break;
            }
        });
        return result;
    };
})(jse.libs.fallback);
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImZhbGxiYWNrLmpzIl0sIm5hbWVzIjpbImpzZSIsImxpYnMiLCJmYWxsYmFjayIsImV4cG9ydHMiLCIkIiwiZXhwciIsInBzZXVkb3MiLCJhdHRyIiwidW5kZWZpbmVkIiwiY3JlYXRlUHNldWRvIiwic2VsZWN0b3IiLCJyZWdleHAiLCJSZWdFeHAiLCJlbGVtIiwiaSIsImF0dHJpYnV0ZXMiLCJsZW5ndGgiLCJ0ZXN0IiwibmFtZSIsIl93YXJuIiwiZnVuY3Rpb25OYW1lIiwiY29yZSIsImRlYnVnIiwid2FybiIsIl9kYXRhIiwiJGVsZW1lbnQiLCJtb2R1bGVOYW1lIiwiaW5pdGlhbERhdGEiLCJkYXRhIiwiZmlsdGVyZWREYXRhIiwiZWFjaCIsImtleSIsInZhbHVlIiwiaW5kZXhPZiIsInRvTG93ZXJDYXNlIiwibmV3S2V5Iiwic3Vic3RyIiwic2V0dXBXaWRnZXRBdHRyIiwiZmlsdGVyIiwiYWRkIiwiZmluZCIsIiRzZWxmIiwibWF0Y2hlZEF0dHJpYnV0ZSIsIm5hbWVzcGFjZU5hbWUiLCJpbmRleCIsImF0dHJpYnV0ZSIsIm1hdGNoIiwicmVwbGFjZSIsImdldFVybFBhcmFtcyIsInVybCIsImRlZXAiLCJkZWNvZGVVUklDb21wb25lbnQiLCJsb2NhdGlvbiIsImhyZWYiLCJzcGxpdFVybCIsInNwbGl0Iiwic3BsaXRQYXJhbSIsInJlZ2V4IiwicmVzdWx0IiwidiIsImtleVZhbHVlIiwicmVnZXhSZXN1bHQiLCJleGVjIiwiYmFzZSIsImJhc2VuYW1lIiwic3Vic3RyaW5nIiwic2VhcmNoIiwia2V5cyIsImxhc3RLZXkiLCJwdXNoIiwibmV4dCIsImdldERhdGEiLCIkZm9ybSIsImlnbm9yZSIsIiRlbGVtZW50cyIsInR5cGUiLCJwcm9wIiwiJHNlbGVjdGVkIiwidmFsIl0sIm1hcHBpbmdzIjoiOztBQUFBOzs7Ozs7Ozs7O0FBVUFBLElBQUlDLElBQUosQ0FBU0MsUUFBVCxHQUFvQkYsSUFBSUMsSUFBSixDQUFTQyxRQUFULElBQXFCLEVBQXpDOztBQUVBOzs7Ozs7Ozs7QUFTQSxDQUFDLFVBQVVDLE9BQVYsRUFBbUI7O0FBRWhCOztBQUVBOzs7Ozs7OztBQU9BLFFBQUlDLEVBQUVDLElBQUYsQ0FBT0MsT0FBUCxDQUFlQyxJQUFmLEtBQXdCQyxTQUE1QixFQUF1QztBQUNuQ0osVUFBRUMsSUFBRixDQUFPQyxPQUFQLENBQWVDLElBQWYsR0FBc0JILEVBQUVDLElBQUYsQ0FBT0ksWUFBUCxDQUFvQixVQUFVQyxRQUFWLEVBQW9CO0FBQzFELGdCQUFJQyxTQUFTLElBQUlDLE1BQUosQ0FBV0YsUUFBWCxDQUFiO0FBQ0EsbUJBQU8sVUFBVUcsSUFBVixFQUFnQjtBQUNuQixxQkFBSyxJQUFJQyxJQUFJLENBQWIsRUFBZ0JBLElBQUlELEtBQUtFLFVBQUwsQ0FBZ0JDLE1BQXBDLEVBQTRDRixHQUE1QyxFQUFpRDtBQUM3Qyx3QkFBSVAsT0FBT00sS0FBS0UsVUFBTCxDQUFnQkQsQ0FBaEIsQ0FBWDtBQUNBLHdCQUFJSCxPQUFPTSxJQUFQLENBQVlWLEtBQUtXLElBQWpCLENBQUosRUFBNEI7QUFDeEIsK0JBQU8sSUFBUDtBQUNIO0FBQ0o7QUFDRCx1QkFBTyxLQUFQO0FBQ0gsYUFSRDtBQVNILFNBWHFCLENBQXRCO0FBWUg7O0FBRUQ7Ozs7Ozs7Ozs7O0FBV0EsYUFBU0MsS0FBVCxDQUFlQyxZQUFmLEVBQTZCO0FBQ3pCcEIsWUFBSXFCLElBQUosQ0FBU0MsS0FBVCxDQUFlQyxJQUFmLENBQW9CLHVCQUFxQkgsWUFBckIsd0VBQXBCO0FBRUg7O0FBRUQ7Ozs7Ozs7O0FBUUFqQixZQUFRcUIsS0FBUixHQUFnQixVQUFVQyxRQUFWLEVBQW9CQyxVQUFwQixFQUFnQztBQUM1Q1AsY0FBTSxPQUFOOztBQUVBLFlBQUlRLGNBQWNGLFNBQVNHLElBQVQsRUFBbEI7QUFBQSxZQUNJQyxlQUFlLEVBRG5COztBQUdBO0FBQ0E7QUFDQXpCLFVBQUUwQixJQUFGLENBQU9ILFdBQVAsRUFBb0IsVUFBQ0ksR0FBRCxFQUFNQyxLQUFOLEVBQWdCO0FBQ2hDLGdCQUFJRCxJQUFJRSxPQUFKLENBQVlQLFVBQVosTUFBNEIsQ0FBNUIsSUFBaUNLLElBQUlFLE9BQUosQ0FBWVAsV0FBV1EsV0FBWCxFQUFaLE1BQTBDLENBQS9FLEVBQWtGO0FBQzlFLG9CQUFJQyxTQUFTSixJQUFJSyxNQUFKLENBQVdWLFdBQVdWLE1BQXRCLENBQWI7QUFDQW1CLHlCQUFTQSxPQUFPQyxNQUFQLENBQWMsQ0FBZCxFQUFpQixDQUFqQixFQUFvQkYsV0FBcEIsS0FBb0NDLE9BQU9DLE1BQVAsQ0FBYyxDQUFkLENBQTdDO0FBQ0FQLDZCQUFhTSxNQUFiLElBQXVCSCxLQUF2QjtBQUNIO0FBQ0osU0FORDs7QUFRQSxlQUFPSCxZQUFQO0FBQ0gsS0FqQkQ7O0FBbUJBOzs7OztBQUtBMUIsWUFBUWtDLGVBQVIsR0FBMEIsVUFBVVosUUFBVixFQUFvQjtBQUMxQ04sY0FBTSxpQkFBTjs7QUFFQU0saUJBQ0thLE1BREwsQ0FDWSw4REFEWixFQUVLQyxHQUZMLENBRVNkLFNBQVNlLElBQVQsQ0FBYyw4REFBZCxDQUZULEVBR0tWLElBSEwsQ0FHVSxZQUFZO0FBQ2QsZ0JBQUlXLFFBQVFyQyxFQUFFLElBQUYsQ0FBWjtBQUFBLGdCQUNJVyxhQUFhMEIsTUFBTSxDQUFOLEVBQVMxQixVQUQxQjtBQUFBLGdCQUVJMkIseUJBRko7QUFBQSxnQkFHSUMsc0JBSEo7O0FBS0F2QyxjQUFFMEIsSUFBRixDQUFPZixVQUFQLEVBQW1CLFVBQVU2QixLQUFWLEVBQWlCQyxTQUFqQixFQUE0QjtBQUMzQyxvQkFBSUEsY0FBY3JDLFNBQWxCLEVBQTZCO0FBQ3pCLDJCQUFPLElBQVAsQ0FEeUIsQ0FDWjtBQUNoQjs7QUFFRGtDLG1DQUFtQkcsVUFBVTNCLElBQVYsQ0FBZTRCLEtBQWYsQ0FBcUIsMkJBQXJCLENBQW5COztBQUVBLG9CQUFJSixxQkFBcUIsSUFBckIsSUFBNkJBLGlCQUFpQjFCLE1BQWpCLEdBQTBCLENBQTNELEVBQThEO0FBQzFEMkIsb0NBQWdCRCxpQkFBaUIsQ0FBakIsRUFBb0JJLEtBQXBCLENBQTBCLGtCQUExQixFQUE4QyxDQUE5QyxDQUFoQjs7QUFFQUwsMEJBQ0tsQyxJQURMLENBQ1VzQyxVQUFVM0IsSUFBVixDQUFlNkIsT0FBZixDQUF1QixVQUFVSixhQUFWLEdBQTBCLElBQWpELEVBQ0YsVUFBVUEsYUFBVixHQUEwQixHQUR4QixDQURWLEVBRXdDRSxVQUFVYixLQUZsRDtBQUdIO0FBQ0osYUFkRDtBQWVILFNBeEJMO0FBeUJILEtBNUJEOztBQThCQTs7Ozs7Ozs7QUFRQTdCLFlBQVE2QyxZQUFSLEdBQXVCLFVBQVVDLEdBQVYsRUFBZUMsSUFBZixFQUFxQjtBQUN4Qy9CLGNBQU0sY0FBTjs7QUFFQThCLGNBQU1FLG1CQUFtQkYsT0FBT0csU0FBU0MsSUFBbkMsQ0FBTjs7QUFFQSxZQUFJQyxXQUFXTCxJQUFJTSxLQUFKLENBQVUsR0FBVixDQUFmO0FBQUEsWUFDSUMsYUFBY0YsU0FBU3RDLE1BQVQsR0FBa0IsQ0FBbkIsR0FBd0JzQyxTQUFTLENBQVQsRUFBWUMsS0FBWixDQUFrQixHQUFsQixDQUF4QixHQUFpRCxFQURsRTtBQUFBLFlBRUlFLFFBQVEsSUFBSTdDLE1BQUosQ0FBVyxZQUFYLENBRlo7QUFBQSxZQUdJOEMsU0FBUyxFQUhiOztBQUtBdEQsVUFBRTBCLElBQUYsQ0FBTzBCLFVBQVAsRUFBbUIsVUFBVTFDLENBQVYsRUFBYTZDLENBQWIsRUFBZ0I7QUFDL0IsZ0JBQUlDLFdBQVdELEVBQUVKLEtBQUYsQ0FBUSxHQUFSLENBQWY7QUFBQSxnQkFDSU0sY0FBY0osTUFBTUssSUFBTixDQUFXRixTQUFTLENBQVQsQ0FBWCxDQURsQjtBQUFBLGdCQUVJRyxPQUFPLElBRlg7QUFBQSxnQkFHSUMsV0FBV0osU0FBUyxDQUFULEVBQVlLLFNBQVosQ0FBc0IsQ0FBdEIsRUFBeUJMLFNBQVMsQ0FBVCxFQUFZTSxNQUFaLENBQW1CLEtBQW5CLENBQXpCLENBSGY7QUFBQSxnQkFJSUMsT0FBTyxFQUpYO0FBQUEsZ0JBS0lDLFVBQVUsSUFMZDs7QUFPQSxnQkFBSSxDQUFDbEIsSUFBRCxJQUFTVyxnQkFBZ0IsSUFBN0IsRUFBbUM7QUFDL0JILHVCQUFPRSxTQUFTLENBQVQsQ0FBUCxJQUFzQkEsU0FBUyxDQUFULEVBQVlMLEtBQVosQ0FBa0IsR0FBbEIsRUFBdUIsQ0FBdkIsQ0FBdEI7QUFDSCxhQUZELE1BRU87O0FBRUhHLHVCQUFPTSxRQUFQLElBQW1CTixPQUFPTSxRQUFQLEtBQW9CLEVBQXZDO0FBQ0FELHVCQUFPTCxPQUFPTSxRQUFQLENBQVA7O0FBRUEsbUJBQUc7QUFDQ0cseUJBQUtFLElBQUwsQ0FBVVIsWUFBWSxDQUFaLENBQVY7QUFDQUEsa0NBQWNKLE1BQU1LLElBQU4sQ0FBV0YsU0FBUyxDQUFULENBQVgsQ0FBZDtBQUNILGlCQUhELFFBR1NDLGdCQUFnQixJQUh6Qjs7QUFLQXpELGtCQUFFMEIsSUFBRixDQUFPcUMsSUFBUCxFQUFhLFVBQVVyRCxDQUFWLEVBQWE2QyxDQUFiLEVBQWdCO0FBQ3pCLHdCQUFJVyxPQUFPSCxLQUFLckQsSUFBSSxDQUFULENBQVg7QUFDQTZDLHdCQUFJQSxLQUFLLEdBQVQ7O0FBRUEsd0JBQUksT0FBUVcsSUFBUixLQUFrQixRQUF0QixFQUFnQztBQUM1QlAsNkJBQUtKLENBQUwsSUFBVUksS0FBS0osQ0FBTCxLQUFXLEVBQXJCO0FBQ0FJLCtCQUFPQSxLQUFLSixDQUFMLENBQVA7QUFDSCxxQkFIRCxNQUdPO0FBQ0hJLDZCQUFLSixDQUFMLElBQVVJLEtBQUtKLENBQUwsS0FBV25ELFNBQXJCO0FBQ0E0RCxrQ0FBVVQsQ0FBVjtBQUNIO0FBQ0osaUJBWEQ7O0FBYUEsb0JBQUlTLFlBQVksSUFBaEIsRUFBc0I7QUFDbEJMLHlCQUFLSyxPQUFMLElBQWdCUixTQUFTLENBQVQsQ0FBaEI7QUFDSCxpQkFGRCxNQUVPO0FBQ0hHLDJCQUFPSCxTQUFTLENBQVQsQ0FBUDtBQUNIO0FBQ0o7QUFFSixTQXhDRDs7QUEwQ0EsZUFBT0YsTUFBUDtBQUNILEtBckREOztBQXVEQTs7Ozs7Ozs7Ozs7QUFXQXZELFlBQVFvRSxPQUFSLEdBQWtCLFVBQVVDLEtBQVYsRUFBaUJDLE1BQWpCLEVBQXlCO0FBQ3ZDdEQsY0FBTSxTQUFOOztBQUVBLFlBQUl1RCxZQUFZRixNQUFNaEMsSUFBTixDQUFXLHlCQUFYLENBQWhCO0FBQUEsWUFDSWtCLFNBQVMsRUFEYjs7QUFHQSxZQUFJZSxNQUFKLEVBQVk7QUFDUkMsd0JBQVlBLFVBQVVwQyxNQUFWLENBQWlCLFVBQVVtQyxNQUFWLEdBQW1CLEdBQXBDLENBQVo7QUFDSDs7QUFFREMsa0JBQVU1QyxJQUFWLENBQWUsWUFBWTtBQUN2QixnQkFBSVcsUUFBUXJDLEVBQUUsSUFBRixDQUFaO0FBQUEsZ0JBQ0l1RSxPQUFPbEMsTUFBTW1DLElBQU4sQ0FBVyxTQUFYLEVBQXNCMUMsV0FBdEIsRUFEWDtBQUFBLGdCQUVJaEIsT0FBT3VCLE1BQU1sQyxJQUFOLENBQVcsTUFBWCxDQUZYO0FBQUEsZ0JBR0lzRSxZQUFZLElBSGhCOztBQUtBRixtQkFBUUEsU0FBUyxPQUFWLEdBQXFCQSxJQUFyQixHQUE0QmxDLE1BQU1sQyxJQUFOLENBQVcsTUFBWCxFQUFtQjJCLFdBQW5CLEVBQW5DOztBQUVBLG9CQUFReUMsSUFBUjtBQUNJLHFCQUFLLE9BQUw7QUFDSUgsMEJBQ0toQyxJQURMLENBQ1UsaUJBQWlCdEIsSUFBakIsR0FBd0IsWUFEbEMsRUFFSzRELEdBRkw7QUFHQTtBQUNKLHFCQUFLLFVBQUw7QUFDSSx3QkFBSTVELEtBQUtnRCxNQUFMLENBQVksS0FBWixNQUF1QixDQUFDLENBQTVCLEVBQStCO0FBQzNCLDRCQUFJekIsTUFBTW1DLElBQU4sQ0FBVyxTQUFYLENBQUosRUFBMkI7QUFDdkIxRCxtQ0FBT0EsS0FBSytDLFNBQUwsQ0FBZSxDQUFmLEVBQWtCL0MsS0FBS2dELE1BQUwsQ0FBWSxLQUFaLENBQWxCLENBQVA7QUFDQSxnQ0FBSVIsT0FBT3hDLElBQVAsTUFBaUJWLFNBQXJCLEVBQWdDO0FBQzVCa0QsdUNBQU94QyxJQUFQLElBQWUsRUFBZjtBQUNIO0FBQ0R3QyxtQ0FBT3hDLElBQVAsRUFBYW1ELElBQWIsQ0FBa0JqRSxFQUFFLElBQUYsRUFBUTBFLEdBQVIsRUFBbEI7QUFDSDtBQUNKLHFCQVJELE1BUU87QUFDSHBCLCtCQUFPeEMsSUFBUCxJQUFldUIsTUFBTW1DLElBQU4sQ0FBVyxTQUFYLENBQWY7QUFDSDtBQUNEO0FBQ0oscUJBQUssUUFBTDtBQUNJQyxnQ0FBWXBDLE1BQU1ELElBQU4sQ0FBVyxXQUFYLENBQVo7QUFDQSx3QkFBSXFDLFVBQVU3RCxNQUFWLEdBQW1CLENBQXZCLEVBQTBCO0FBQ3RCMEMsK0JBQU94QyxJQUFQLElBQWUsRUFBZjtBQUNBMkQsa0NBQVUvQyxJQUFWLENBQWUsWUFBWTtBQUN2QjRCLG1DQUFPeEMsSUFBUCxFQUFhbUQsSUFBYixDQUFrQmpFLEVBQUUsSUFBRixFQUFRMEUsR0FBUixFQUFsQjtBQUNILHlCQUZEO0FBR0gscUJBTEQsTUFLTztBQUNIcEIsK0JBQU94QyxJQUFQLElBQWUyRCxVQUFVQyxHQUFWLEVBQWY7QUFDSDtBQUNEO0FBQ0oscUJBQUssUUFBTDtBQUNJO0FBQ0o7QUFDSSx3QkFBSTVELElBQUosRUFBVTtBQUNOd0MsK0JBQU94QyxJQUFQLElBQWV1QixNQUFNcUMsR0FBTixFQUFmO0FBQ0g7QUFDRDtBQXBDUjtBQXNDSCxTQTlDRDtBQStDQSxlQUFPcEIsTUFBUDtBQUNILEtBMUREO0FBNERILENBOU9ELEVBOE9HMUQsSUFBSUMsSUFBSixDQUFTQyxRQTlPWiIsImZpbGUiOiJmYWxsYmFjay5qcyIsInNvdXJjZXNDb250ZW50IjpbIi8qIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiBmYWxsYmFjay5qcyAyMDE2LTA5LTA4XHJcbiBHYW1iaW8gR21iSFxyXG4gaHR0cDovL3d3dy5nYW1iaW8uZGVcclxuIENvcHlyaWdodCAoYykgMjAxNiBHYW1iaW8gR21iSFxyXG4gUmVsZWFzZWQgdW5kZXIgdGhlIEdOVSBHZW5lcmFsIFB1YmxpYyBMaWNlbnNlIChWZXJzaW9uIDIpXHJcbiBbaHR0cDovL3d3dy5nbnUub3JnL2xpY2Vuc2VzL2dwbC0yLjAuaHRtbF1cclxuIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAqL1xyXG5cclxuanNlLmxpYnMuZmFsbGJhY2sgPSBqc2UubGlicy5mYWxsYmFjayB8fCB7fTtcclxuXHJcbi8qKlxyXG4gKiAjIyBGYWxsYmFjayBMaWJyYXJ5XHJcbiAqXHJcbiAqIFRoaXMgbGlicmFyeSBjb250YWlucyBhIHNldCBvZiBkZXByZWNhdGVkIGZ1bmN0aW9ucyB0aGF0IGFyZSBzdGlsbCBwcmVzZW50IGZvciBmYWxsYmFjayBzdXBwb3J0LiBEbyBub3RcclxuICogdXNlIHRoZXNlIG1ldGhvZHMgaW4gbmV3IG1vZHVsZXMuXHJcbiAqXHJcbiAqIEBtb2R1bGUgSlNFL0xpYnMvZmFsbGJhY2tcclxuICogQGV4cG9ydHMganNlLmxpYnMuZmFsbGJhY2tcclxuICovXHJcbihmdW5jdGlvbiAoZXhwb3J0cykge1xyXG5cclxuICAgICd1c2Ugc3RyaWN0JztcclxuXHJcbiAgICAvKipcclxuICAgICAqIEFkZCBcIjphdHRyXCIgcHNldWRvIHNlbGVjdG9yLlxyXG4gICAgICpcclxuICAgICAqIFRoaXMgcHNldWRvIHNlbGVjdG9yIGlzIG5vcm1hbGx5IGVuYWJsZWQgYnkgaW5jbHVkaW5nIHRoZSBKU0VuZ2luZSBcImpxdWVyeV9leHRlbnNpb25zXCIgbGlicmFyeS4gSG9uZXlncmlkXHJcbiAgICAgKiB0aHJvdWdoIG5lZWRzIHRoaXMgcHNldWRvIHNlbGVjdG9yIGluIHRoaXMgbGlicmFyeSB3aGljaCBtaWdodCBiZSBsb2FkZWQgcHJpb3IgdG8ganF1ZXJ5X2V4dGVuc2lvbnMgYW5kXHJcbiAgICAgKiB0aGlzIGlzIHdoeSB3ZSBkZWZpbmUgaXQgb25jZSBhZ2FpbiBpbiB0aGlzIGZpbGUuXHJcbiAgICAgKi9cclxuICAgIGlmICgkLmV4cHIucHNldWRvcy5hdHRyID09PSB1bmRlZmluZWQpIHtcclxuICAgICAgICAkLmV4cHIucHNldWRvcy5hdHRyID0gJC5leHByLmNyZWF0ZVBzZXVkbyhmdW5jdGlvbiAoc2VsZWN0b3IpIHtcclxuICAgICAgICAgICAgbGV0IHJlZ2V4cCA9IG5ldyBSZWdFeHAoc2VsZWN0b3IpO1xyXG4gICAgICAgICAgICByZXR1cm4gZnVuY3Rpb24gKGVsZW0pIHtcclxuICAgICAgICAgICAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgZWxlbS5hdHRyaWJ1dGVzLmxlbmd0aDsgaSsrKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgbGV0IGF0dHIgPSBlbGVtLmF0dHJpYnV0ZXNbaV07XHJcbiAgICAgICAgICAgICAgICAgICAgaWYgKHJlZ2V4cC50ZXN0KGF0dHIubmFtZSkpIHtcclxuICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHRydWU7XHJcbiAgICAgICAgICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlO1xyXG4gICAgICAgICAgICB9O1xyXG4gICAgICAgIH0pO1xyXG4gICAgfVxyXG5cclxuICAgIC8qKlxyXG4gICAgICogQWRkIGEgZmFsbGJhY2sgdXNhZ2Ugd2FybmluZyBpbiB0aGUgY29uc29sZS5cclxuICAgICAqXHJcbiAgICAgKiBBcyB0aGUgSlMgZW5naW5lIGV2b2x2ZXMgbWFueSBvbGQgZmVhdHVyZXMgd2lsbCBuZWVkIHRvIGJlIGNoYW5nZWQgaW4gb3JkZXIgdG8gbGV0IGEgZmluZXIgYW5kIGNsZWFyZXJcclxuICAgICAqIEFQSSBmb3IgdGhlIEpTIEVuZ2luZSBjb3JlIG1lY2hhbmlzbXMuIFVzZSB0aGlzIG1ldGhvZCB0byBjcmVhdGUgYSBmYWxsYmFjayB1c2FnZSB3YXJuaW5nIGZvciB0aGUgZnVuY3Rpb25zXHJcbiAgICAgKiBwbGFjZWQgd2l0aGluIHRoaXMgbGlicmFyeS5cclxuICAgICAqXHJcbiAgICAgKiBAcGFyYW0ge1N0cmluZ30gZnVuY3Rpb25OYW1lIFRoZSBkZXByZWNhdGVkIGZ1bmN0aW9uIG5hbWUuXHJcbiAgICAgKlxyXG4gICAgICogQHByaXZhdGVcclxuICAgICAqL1xyXG4gICAgZnVuY3Rpb24gX3dhcm4oZnVuY3Rpb25OYW1lKSB7XHJcbiAgICAgICAganNlLmNvcmUuZGVidWcud2FybihganNlLmxpYnMuZmFsbGJhY2suJHtmdW5jdGlvbk5hbWV9IHdhcyBjYWxsZWQhIGBcclxuICAgICAgICAgICAgKyBgQXZvaWQgdGhlIHVzZSBvZiBmYWxsYmFjayBtZXRob2RzIGluIG5ldyBtb2R1bGVzLmApO1xyXG4gICAgfVxyXG5cclxuICAgIC8qKlxyXG4gICAgICogR2V0IHRoZSBtb2R1bGUgcmVsYXRlZCBkYXRhIG9mIHRoZSBwcm92aWRlZCBlbGVtZW50LlxyXG4gICAgICpcclxuICAgICAqIEBwYXJhbSB7alF1ZXJ5fSAkZWxlbWVudFxyXG4gICAgICogQHBhcmFtIHtTdHJpbmd9IG1vZHVsZU5hbWVcclxuICAgICAqXHJcbiAgICAgKiBAcmV0dXJuIHtPYmplY3R9XHJcbiAgICAgKi9cclxuICAgIGV4cG9ydHMuX2RhdGEgPSBmdW5jdGlvbiAoJGVsZW1lbnQsIG1vZHVsZU5hbWUpIHtcclxuICAgICAgICBfd2FybignX2RhdGEnKTtcclxuXHJcbiAgICAgICAgbGV0IGluaXRpYWxEYXRhID0gJGVsZW1lbnQuZGF0YSgpLFxyXG4gICAgICAgICAgICBmaWx0ZXJlZERhdGEgPSB7fTtcclxuXHJcbiAgICAgICAgLy8gU2VhcmNoZXMgZm9yIG1vZHVsZSByZWxldmFudCBkYXRhIGluc2lkZSB0aGUgbWFpbi1kYXRhLW9iamVjdC5cclxuICAgICAgICAvLyBEYXRhIGZvciBvdGhlciB3aWRnZXRzIHdpbGwgbm90IGdldCBwYXNzZWQgdG8gdGhpcyB3aWRnZXRcclxuICAgICAgICAkLmVhY2goaW5pdGlhbERhdGEsIChrZXksIHZhbHVlKSA9PiB7XHJcbiAgICAgICAgICAgIGlmIChrZXkuaW5kZXhPZihtb2R1bGVOYW1lKSA9PT0gMCB8fCBrZXkuaW5kZXhPZihtb2R1bGVOYW1lLnRvTG93ZXJDYXNlKCkpID09PSAwKSB7XHJcbiAgICAgICAgICAgICAgICBsZXQgbmV3S2V5ID0ga2V5LnN1YnN0cihtb2R1bGVOYW1lLmxlbmd0aCk7XHJcbiAgICAgICAgICAgICAgICBuZXdLZXkgPSBuZXdLZXkuc3Vic3RyKDAsIDEpLnRvTG93ZXJDYXNlKCkgKyBuZXdLZXkuc3Vic3RyKDEpO1xyXG4gICAgICAgICAgICAgICAgZmlsdGVyZWREYXRhW25ld0tleV0gPSB2YWx1ZTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgIH0pO1xyXG5cclxuICAgICAgICByZXR1cm4gZmlsdGVyZWREYXRhO1xyXG4gICAgfTtcclxuXHJcbiAgICAvKipcclxuICAgICAqIFNldHVwIFdpZGdldCBBdHRyaWJ1dGVcclxuICAgICAqXHJcbiAgICAgKiBAcGFyYW0ge09iamVjdH0gJGVsZW1lbnQgQ2hhbmdlIHRoZSB3aWRnZXQgYXR0cmlidXRlIG9mIGFuIGVsZW1lbnQuXHJcbiAgICAgKi9cclxuICAgIGV4cG9ydHMuc2V0dXBXaWRnZXRBdHRyID0gZnVuY3Rpb24gKCRlbGVtZW50KSB7XHJcbiAgICAgICAgX3dhcm4oJ3NldHVwV2lkZ2V0QXR0cicpO1xyXG5cclxuICAgICAgICAkZWxlbWVudFxyXG4gICAgICAgICAgICAuZmlsdGVyKCc6YXR0ciheZGF0YS1neC1fKSwgOmF0dHIoXmRhdGEtZ2FtYmlvLV8pLCA6YXR0ciheZGF0YS1qc2UtXyknKVxyXG4gICAgICAgICAgICAuYWRkKCRlbGVtZW50LmZpbmQoJzphdHRyKF5kYXRhLWd4LV8pLCA6YXR0ciheZGF0YS1nYW1iaW8tXyksIDphdHRyKF5kYXRhLWpzZS1fKScpKVxyXG4gICAgICAgICAgICAuZWFjaChmdW5jdGlvbiAoKSB7XHJcbiAgICAgICAgICAgICAgICBsZXQgJHNlbGYgPSAkKHRoaXMpLFxyXG4gICAgICAgICAgICAgICAgICAgIGF0dHJpYnV0ZXMgPSAkc2VsZlswXS5hdHRyaWJ1dGVzLFxyXG4gICAgICAgICAgICAgICAgICAgIG1hdGNoZWRBdHRyaWJ1dGUsXHJcbiAgICAgICAgICAgICAgICAgICAgbmFtZXNwYWNlTmFtZTtcclxuXHJcbiAgICAgICAgICAgICAgICAkLmVhY2goYXR0cmlidXRlcywgZnVuY3Rpb24gKGluZGV4LCBhdHRyaWJ1dGUpIHtcclxuICAgICAgICAgICAgICAgICAgICBpZiAoYXR0cmlidXRlID09PSB1bmRlZmluZWQpIHtcclxuICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHRydWU7IC8vIHdyb25nIGF0dHJpYnV0ZSwgY29udGludWUgbG9vcFxyXG4gICAgICAgICAgICAgICAgICAgIH1cclxuXHJcbiAgICAgICAgICAgICAgICAgICAgbWF0Y2hlZEF0dHJpYnV0ZSA9IGF0dHJpYnV0ZS5uYW1lLm1hdGNoKC9kYXRhLShnYW1iaW98Z3h8anNlKS1fLiovZyk7XHJcblxyXG4gICAgICAgICAgICAgICAgICAgIGlmIChtYXRjaGVkQXR0cmlidXRlICE9PSBudWxsICYmIG1hdGNoZWRBdHRyaWJ1dGUubGVuZ3RoID4gMCkge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICBuYW1lc3BhY2VOYW1lID0gbWF0Y2hlZEF0dHJpYnV0ZVswXS5tYXRjaCgvKGdhbWJpb3xneHxqc2UpL2cpWzBdO1xyXG5cclxuICAgICAgICAgICAgICAgICAgICAgICAgJHNlbGZcclxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIC5hdHRyKGF0dHJpYnV0ZS5uYW1lLnJlcGxhY2UoJ2RhdGEtJyArIG5hbWVzcGFjZU5hbWUgKyAnLV8nLFxyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdkYXRhLScgKyBuYW1lc3BhY2VOYW1lICsgJy0nKSwgYXR0cmlidXRlLnZhbHVlKTtcclxuICAgICAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgICB9KTtcclxuICAgICAgICAgICAgfSk7XHJcbiAgICB9O1xyXG5cclxuICAgIC8qKlxyXG4gICAgICogR2V0IFVSTCBwYXJhbWV0ZXJzLlxyXG4gICAgICpcclxuICAgICAqIEBwYXJhbSB7U3RyaW5nfSB1cmxcclxuICAgICAqIEBwYXJhbSB7Qm9vbGVhbn0gZGVlcFxyXG4gICAgICpcclxuICAgICAqIEByZXR1cm4ge09iamVjdH1cclxuICAgICAqL1xyXG4gICAgZXhwb3J0cy5nZXRVcmxQYXJhbXMgPSBmdW5jdGlvbiAodXJsLCBkZWVwKSB7XHJcbiAgICAgICAgX3dhcm4oJ2dldFVybFBhcmFtcycpO1xyXG5cclxuICAgICAgICB1cmwgPSBkZWNvZGVVUklDb21wb25lbnQodXJsIHx8IGxvY2F0aW9uLmhyZWYpO1xyXG5cclxuICAgICAgICBsZXQgc3BsaXRVcmwgPSB1cmwuc3BsaXQoJz8nKSxcclxuICAgICAgICAgICAgc3BsaXRQYXJhbSA9IChzcGxpdFVybC5sZW5ndGggPiAxKSA/IHNwbGl0VXJsWzFdLnNwbGl0KCcmJykgOiBbXSxcclxuICAgICAgICAgICAgcmVnZXggPSBuZXcgUmVnRXhwKC9cXFsoLio/KVxcXS9nKSxcclxuICAgICAgICAgICAgcmVzdWx0ID0ge307XHJcblxyXG4gICAgICAgICQuZWFjaChzcGxpdFBhcmFtLCBmdW5jdGlvbiAoaSwgdikge1xyXG4gICAgICAgICAgICBsZXQga2V5VmFsdWUgPSB2LnNwbGl0KCc9JyksXHJcbiAgICAgICAgICAgICAgICByZWdleFJlc3VsdCA9IHJlZ2V4LmV4ZWMoa2V5VmFsdWVbMF0pLFxyXG4gICAgICAgICAgICAgICAgYmFzZSA9IG51bGwsXHJcbiAgICAgICAgICAgICAgICBiYXNlbmFtZSA9IGtleVZhbHVlWzBdLnN1YnN0cmluZygwLCBrZXlWYWx1ZVswXS5zZWFyY2goJ1xcXFxbJykpLFxyXG4gICAgICAgICAgICAgICAga2V5cyA9IFtdLFxyXG4gICAgICAgICAgICAgICAgbGFzdEtleSA9IG51bGw7XHJcblxyXG4gICAgICAgICAgICBpZiAoIWRlZXAgfHwgcmVnZXhSZXN1bHQgPT09IG51bGwpIHtcclxuICAgICAgICAgICAgICAgIHJlc3VsdFtrZXlWYWx1ZVswXV0gPSBrZXlWYWx1ZVsxXS5zcGxpdCgnIycpWzBdO1xyXG4gICAgICAgICAgICB9IGVsc2Uge1xyXG5cclxuICAgICAgICAgICAgICAgIHJlc3VsdFtiYXNlbmFtZV0gPSByZXN1bHRbYmFzZW5hbWVdIHx8IFtdO1xyXG4gICAgICAgICAgICAgICAgYmFzZSA9IHJlc3VsdFtiYXNlbmFtZV07XHJcblxyXG4gICAgICAgICAgICAgICAgZG8ge1xyXG4gICAgICAgICAgICAgICAgICAgIGtleXMucHVzaChyZWdleFJlc3VsdFsxXSk7XHJcbiAgICAgICAgICAgICAgICAgICAgcmVnZXhSZXN1bHQgPSByZWdleC5leGVjKGtleVZhbHVlWzBdKTtcclxuICAgICAgICAgICAgICAgIH0gd2hpbGUgKHJlZ2V4UmVzdWx0ICE9PSBudWxsKTtcclxuXHJcbiAgICAgICAgICAgICAgICAkLmVhY2goa2V5cywgZnVuY3Rpb24gKGksIHYpIHtcclxuICAgICAgICAgICAgICAgICAgICBsZXQgbmV4dCA9IGtleXNbaSArIDFdO1xyXG4gICAgICAgICAgICAgICAgICAgIHYgPSB2IHx8ICcwJztcclxuXHJcbiAgICAgICAgICAgICAgICAgICAgaWYgKHR5cGVvZiAobmV4dCkgPT09ICdzdHJpbmcnKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgIGJhc2Vbdl0gPSBiYXNlW3ZdIHx8IFtdO1xyXG4gICAgICAgICAgICAgICAgICAgICAgICBiYXNlID0gYmFzZVt2XTtcclxuICAgICAgICAgICAgICAgICAgICB9IGVsc2Uge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICBiYXNlW3ZdID0gYmFzZVt2XSB8fCB1bmRlZmluZWQ7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgIGxhc3RLZXkgPSB2O1xyXG4gICAgICAgICAgICAgICAgICAgIH1cclxuICAgICAgICAgICAgICAgIH0pO1xyXG5cclxuICAgICAgICAgICAgICAgIGlmIChsYXN0S2V5ICE9PSBudWxsKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgYmFzZVtsYXN0S2V5XSA9IGtleVZhbHVlWzFdO1xyXG4gICAgICAgICAgICAgICAgfSBlbHNlIHtcclxuICAgICAgICAgICAgICAgICAgICBiYXNlID0ga2V5VmFsdWVbMV07XHJcbiAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgIH1cclxuXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIHJldHVybiByZXN1bHQ7XHJcbiAgICB9O1xyXG5cclxuICAgIC8qKlxyXG4gICAgICogRmFsbGJhY2sgZ2V0RGF0YSBtZXRob2QuXHJcbiAgICAgKlxyXG4gICAgICogVGhpcyBtZXRob2Qgd2FzIGluY2x1ZGVkIGluIHYxLjAgb2YgSlMgRW5naW5lIGFuZCBpcyByZXBsYWNlZCBieSB0aGVcclxuICAgICAqIFwianNlLmxpYnMuZm9ybS5nZXREYXRhXCIgbWV0aG9kLlxyXG4gICAgICpcclxuICAgICAqIEBwYXJhbSB7T2JqZWN0fSAkZm9ybSBTZWxlY3RvciBvZiB0aGUgZm9ybSB0byBiZSBwYXJzZWQuXHJcbiAgICAgKiBAcGFyYW0ge1N0cmluZ30gaWdub3JlIChvcHRpb25hbCkgalF1ZXJ5IHNlbGVjdG9yIHN0cmluZyBvZiBmb3JtIGVsZW1lbnRzIHRvIGJlIGlnbm9yZWQuXHJcbiAgICAgKlxyXG4gICAgICogQHJldHVybiB7T2JqZWN0fSBSZXR1cm5zIHRoZSBkYXRhIG9mIHRoZSBmb3JtIGFzIGFuIG9iamVjdC5cclxuICAgICAqL1xyXG4gICAgZXhwb3J0cy5nZXREYXRhID0gZnVuY3Rpb24gKCRmb3JtLCBpZ25vcmUpIHtcclxuICAgICAgICBfd2FybignZ2V0RGF0YScpO1xyXG5cclxuICAgICAgICBsZXQgJGVsZW1lbnRzID0gJGZvcm0uZmluZCgnaW5wdXQsIHRleHRhcmVhLCBzZWxlY3QnKSxcclxuICAgICAgICAgICAgcmVzdWx0ID0ge307XHJcblxyXG4gICAgICAgIGlmIChpZ25vcmUpIHtcclxuICAgICAgICAgICAgJGVsZW1lbnRzID0gJGVsZW1lbnRzLmZpbHRlcignOm5vdCgnICsgaWdub3JlICsgJyknKTtcclxuICAgICAgICB9XHJcblxyXG4gICAgICAgICRlbGVtZW50cy5lYWNoKGZ1bmN0aW9uICgpIHtcclxuICAgICAgICAgICAgbGV0ICRzZWxmID0gJCh0aGlzKSxcclxuICAgICAgICAgICAgICAgIHR5cGUgPSAkc2VsZi5wcm9wKCd0YWdOYW1lJykudG9Mb3dlckNhc2UoKSxcclxuICAgICAgICAgICAgICAgIG5hbWUgPSAkc2VsZi5hdHRyKCduYW1lJyksXHJcbiAgICAgICAgICAgICAgICAkc2VsZWN0ZWQgPSBudWxsO1xyXG5cclxuICAgICAgICAgICAgdHlwZSA9ICh0eXBlICE9PSAnaW5wdXQnKSA/IHR5cGUgOiAkc2VsZi5hdHRyKCd0eXBlJykudG9Mb3dlckNhc2UoKTtcclxuXHJcbiAgICAgICAgICAgIHN3aXRjaCAodHlwZSkge1xyXG4gICAgICAgICAgICAgICAgY2FzZSAncmFkaW8nOlxyXG4gICAgICAgICAgICAgICAgICAgICRmb3JtXHJcbiAgICAgICAgICAgICAgICAgICAgICAgIC5maW5kKCdpbnB1dFtuYW1lPVwiJyArIG5hbWUgKyAnXCJdOmNoZWNrZWQnKVxyXG4gICAgICAgICAgICAgICAgICAgICAgICAudmFsKCk7XHJcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICAgICAgICBjYXNlICdjaGVja2JveCc6XHJcbiAgICAgICAgICAgICAgICAgICAgaWYgKG5hbWUuc2VhcmNoKCdcXFxcWycpICE9PSAtMSkge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICBpZiAoJHNlbGYucHJvcCgnY2hlY2tlZCcpKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBuYW1lID0gbmFtZS5zdWJzdHJpbmcoMCwgbmFtZS5zZWFyY2goJ1xcXFxbJykpO1xyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgaWYgKHJlc3VsdFtuYW1lXSA9PT0gdW5kZWZpbmVkKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgcmVzdWx0W25hbWVdID0gW107XHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICByZXN1bHRbbmFtZV0ucHVzaCgkKHRoaXMpLnZhbCgpKTtcclxuICAgICAgICAgICAgICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICAgICAgICAgIH0gZWxzZSB7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlc3VsdFtuYW1lXSA9ICRzZWxmLnByb3AoJ2NoZWNrZWQnKTtcclxuICAgICAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICAgICAgICBjYXNlICdzZWxlY3QnOlxyXG4gICAgICAgICAgICAgICAgICAgICRzZWxlY3RlZCA9ICRzZWxmLmZpbmQoJzpzZWxlY3RlZCcpO1xyXG4gICAgICAgICAgICAgICAgICAgIGlmICgkc2VsZWN0ZWQubGVuZ3RoID4gMSkge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICByZXN1bHRbbmFtZV0gPSBbXTtcclxuICAgICAgICAgICAgICAgICAgICAgICAgJHNlbGVjdGVkLmVhY2goZnVuY3Rpb24gKCkge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmVzdWx0W25hbWVdLnB1c2goJCh0aGlzKS52YWwoKSk7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgICAgICAgICAgIH0gZWxzZSB7XHJcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlc3VsdFtuYW1lXSA9ICRzZWxlY3RlZC52YWwoKTtcclxuICAgICAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICAgICAgICBjYXNlICdidXR0b24nOlxyXG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgICAgICAgICAgICBpZiAobmFtZSkge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICByZXN1bHRbbmFtZV0gPSAkc2VsZi52YWwoKTtcclxuICAgICAgICAgICAgICAgICAgICB9XHJcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICAgIH1cclxuICAgICAgICB9KTtcclxuICAgICAgICByZXR1cm4gcmVzdWx0O1xyXG4gICAgfTtcclxuXHJcbn0pKGpzZS5saWJzLmZhbGxiYWNrKTsgIl19
